import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getExpensesCollection } from "@/lib/database/collections"
import { calculateSalaryTotals, calculateTaxTotals } from "@/lib/utils/calculations"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const expensesCollection = await getExpensesCollection()
    const expense = await expensesCollection.findOne({ _id: new ObjectId(params.id) })

    if (!expense) {
      return NextResponse.json({ success: false, error: "Expense record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: expense })
  } catch (error) {
    console.error("Error fetching expense:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch expense record" }, { status: 500 })
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body = await request.json()

    // Calculate salary totals if applicable
    const salaryTotals = calculateSalaryTotals({
      hourlyRate: body.hourlyRate,
      totalHours: body.totalHours,
      received: body.received,
    })

    // Calculate tax totals if applicable
    const taxTotals = calculateTaxTotals({
      netto: body.netto,
      vat: body.vat,
    })

    const updateData = {
      date: new Date(body.date),
      branchId: new ObjectId(body.branchId),
      category: body.category,
      subcategory: body.subcategory,
      amount: body.amount,
      description: body.description,
      paymentMethod: body.paymentMethod,
      employeeId: body.employeeId ? new ObjectId(body.employeeId) : undefined,
      hourlyRate: body.hourlyRate,
      totalHours: body.totalHours,
      totalSalary: salaryTotals.totalSalary || body.totalSalary,
      received: body.received,
      payable: salaryTotals.payable || body.payable,
      netto: body.netto,
      vat: body.vat,
      brutto: taxTotals.brutto || body.brutto,
      fakturaNo: body.fakturaNo,
      notes: body.notes,
      paymentDate: body.paymentDate ? new Date(body.paymentDate) : undefined,
      updatedAt: new Date(),
    }

    const expensesCollection = await getExpensesCollection()
    const result = await expensesCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "Expense record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("Error updating expense:", error)
    return NextResponse.json({ success: false, error: "Failed to update expense record" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const expensesCollection = await getExpensesCollection()
    const result = await expensesCollection.deleteOne({ _id: new ObjectId(params.id) })

    if (result.deletedCount === 0) {
      return NextResponse.json({ success: false, error: "Expense record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, message: "Expense record deleted successfully" })
  } catch (error) {
    console.error("Error deleting expense:", error)
    return NextResponse.json({ success: false, error: "Failed to delete expense record" }, { status: 500 })
  }
}
